#include "grain_growth.hpp"
#include "cxxopts.hpp"

#include <ctime>

valueType *read_init_state(const char *filename, uint &Nx, uint &Ny, uint &n_grains)
{
    FILE *inp = fopen(filename, "r");
    fscanf(inp, "%u,%u,%u", &Nx, &Ny, &n_grains);
    valueType *mtx = new valueType[Nx * Ny * n_grains];
    for (uint pg = 0; pg < n_grains; ++pg)
    {
        for (uint x = 0; x < Nx; ++x)
        {
            fscanf(inp, "%lf", mtx + pg * Nx * Ny + x * Ny);
            for (uint y = 1; y < Ny; ++y)
                fscanf(inp, ",%lf", mtx + pg * Nx * Ny + x * Ny + y);
        }
    }

    fclose(inp);
    return mtx;
}

void print_img_in_csv(valueType *img, const char *filename, uint Nx, uint Ny,
                      uint n_grains)
{
    FILE *oup = fopen(filename, "w");
    fprintf(oup, "%u,%u,%u\n", Nx, Ny, n_grains);
    for (uint pg = 0; pg < n_grains; ++pg)
    {
        for (uint i = 0; i < Nx; ++i)
        {
            fprintf(oup, "%lf", img[pg * Nx * Ny + i * Ny]);
            for (uint j = 1; j < Ny; ++j)
            {
                fprintf(oup, ",%lf", img[pg * Nx * Ny + i * Ny + j]);
            }
            fputc('\n', oup);
        }
    }
    fclose(oup);
}

class Args
{
public:
    uint nsteps;
    string input;
    string output;
    string bucket_output;
    uint lshL, lshK; // Nx, Ny;
    double lshr;
};

Args *parse_args(int argc, const char *argv[])
{
    try
    {
        Args *args = new Args;

        cxxopts::Options options(argv[0], " - test forward simulation of grain growth.");
        options
            .positional_help("[optional args]")
            .show_positional_help();

        options
            .set_width(70)
            .set_tab_expansion()
            .allow_unrecognised_options()
            .add_options()("s,nsteps", "Number of steps of simulation (default=100)", cxxopts::value<int>(), "N")("o,output", "Output file (default=grain.out)", cxxopts::value<std::string>(), "FILE")("i,input", "Input file (default=grain.in)", cxxopts::value<std::string>(), "FILE")("lshK", "K for LSH (default=1)", cxxopts::value<int>(), "INT")("lshL", "L for LSH (default=1)", cxxopts::value<int>(), "INT")("lshr", "r for LSH (default=1e-4)", cxxopts::value<float>(), "FLOAT")("bucket_output", "Output file of the bucket information (default=bucket.out)", cxxopts::value<std::string>(), "FILE")("h,help", "Print help")
#ifdef CXXOPTS_USE_UNICODE
                ("unicode", u8"A help option with non-ascii: à. Here the size of the"
                            " string should be correct")
#endif
            ;
        //("Nx", "size of x-axis (default=64)", cxxopts::value<int>(), "INT")
        //("Ny", "size of y-axis (default=64)", cxxopts::value<int>(), "INT")

        auto result = options.parse(argc, argv);

        if (result.count("help"))
        {
            std::cout << options.help({"", "Group"}) << std::endl;
            exit(0);
        }

        std::cout << "[Parse Args]" << std::endl;

        if (result.count("nsteps"))
        {
            std::cout << "  nsteps = " << result["nsteps"].as<int>() << std::endl;
            args->nsteps = (uint)result["nsteps"].as<int>();
        }
        else
        {
            args->nsteps = 100;
        }

        if (result.count("output"))
        {
            std::cout << "  output = " << result["output"].as<std::string>()
                      << std::endl;
            args->output = result["output"].as<std::string>();
        }
        else
        {
            args->output = "grain.out";
        }

        if (result.count("input"))
        {
            std::cout << "  input = " << result["input"].as<std::string>()
                      << std::endl;
            args->input = result["input"].as<std::string>();
        }
        else
        {
            args->input = "grain.in";
        }

        if (result.count("bucket_output"))
        {
            std::cout << "  bucket_output = " << result["bucket_output"].as<std::string>()
                      << std::endl;
            args->bucket_output = result["bucket_output"].as<std::string>();
        }
        else
        {
            args->bucket_output = "bucket.out";
        }
        if (result.count("lshK"))
        {
            std::cout << "  lshK = " << result["lshK"].as<int>()
                      << std::endl;
            args->lshK = (uint)result["lshK"].as<int>();
        }
        else
        {
            args->lshK = 1;
        }

        if (result.count("lshL"))
        {
            std::cout << "  lshL = " << result["lshL"].as<int>()
                      << std::endl;
            args->lshL = (uint)result["lshL"].as<int>();
        }
        else
        {
            args->lshL = 1;
        }

        if (result.count("lshr"))
        {
            std::cout << "  lshr = " << result["lshr"].as<float>()
                      << std::endl;
            args->lshr = (double)result["lshr"].as<float>();
        }
        else
        {
            args->lshr = 1e-4;
        }

        auto arguments = result.arguments();
        std::cout << "  Saw " << arguments.size() << " arguments" << std::endl;

        std::cout << "[End of Parse Args]" << std::endl;

        /*
    if (result.count("Nx"))
    {
      std::cout << "  Nx = " << result["Nx"].as<int>()
        << std::endl;
      args->Nx = (uint)result["Nx"].as<int>();
    }else{
      args->Nx = 64;
    }
    if (result.count("Ny"))
    {
      std::cout << "  Ny = " << result["Ny"].as<int>()
        << std::endl;
      args->Ny = (uint)result["Ny"].as<int>();
    }else{
      args->Ny = 64;
    }
    */

        return args;
    }
    catch (const cxxopts::OptionException &e)
    {
        std::cout << "error parsing options: " << e.what() << std::endl;
        exit(1);
    }
}

int main(int argc, const char *argv[])
{
    Args *args = parse_args(argc, argv);

    // def parameters
    uint Nx = 64; //1024;   these will be changed later.
    uint Ny = 64; //1024;
    uint n_grains = 2;

    uint lshK = args->lshK;
    uint lshL = args->lshL;
    valueType lsh_r = args->lshr;
    uint nsteps = args->nsteps;

    valueType h = 0.5;

    valueType A = 1.0;
    valueType B = 1.0;
    valueType L = 5.0;
    valueType kappa = 0.1;

    valueType dtime = 0.05;
    valueType ttime = 0.0;

    // form initial matrix
    printf("before read_init_state, input=%s\n", args->input.c_str());
    // valueType *mtx = read_init_state("../grain_growth_intuition_1", Nx, Ny, n_grains);

    uint start_skip = 1;
    uint skip_step = 30;
    // uint skip_step = 5;
    char *data_path = "../grain_growth_all_data_1";
    GrainGrowthDataset dataset(data_path, start_skip, skip_step);

    uint n_step = 500;
    Nx = dataset.Nx;
    Ny = dataset.Ny;
    n_grains = dataset.n_grains;
    n_step = dataset.n_step;

    ReturnItem rt = dataset.get_item(0);
    valueType* etas = rt.data.eta1_eta2;
    print_img_in_csv(etas, "../forward_1/ref_start", Nx, Ny, n_grains);
    print_img_in_csv(etas, "../forward_1/sim_start", Nx, Ny, n_grains);
    // form OneStep class
    assert(Nx == Ny);

    printf("  [Nx=%u,Ny=%u,n_grains=%u]\n", Nx, Ny, n_grains);
    printf("before construct one_step\n");
    GrainGrowthOneStep one_step(Nx, Ny, n_grains, lshK, lshL, h,
                                A, B, L, kappa, dtime, lsh_r);
    printf("before encode_from_img\n");
    fflush(stdout);

    //time_t start_time;
    //time(&start_time);

    struct timespec tstart = {0, 0}, tend = {0, 0};
    clock_gettime(CLOCK_MONOTONIC, &tstart);

    one_step.encode_from_img(etas);
    one_step.hash_t.clean_up_l_list();
    printf("finish encode_from_img\n");
    fflush(stdout);

    //  one_step.hash_t.print_hash_table(one_step.inv, args->bucket_output.c_str());
    // forward for 500 step
    nsteps = dataset.get_len();
    for (uint step = 0; step < nsteps;)
    {
        one_step.next();

        ttime += dtime;
        ++step;

        if (step % 10 == 0)
        {
            printf("step=%u, ttime=%lf\n", step, ttime);
            one_step.hash_t.clean_up_l_list();
        }

        if ((step % 30 == 0) || (step==1))
        {
            valueType* result = one_step.decode_to_img();
            std::string sim_filename = std::string("../forward_1/sim_" + std::to_string(step));
            print_img_in_csv(result, sim_filename.c_str(), Nx, Ny, n_grains);
            ReturnItem rf = dataset.get_item(step);
            valueType* ref_result = rf.data.eta1_eta2;
            std::string ref_filename = std::string("../forward_1/ref_" + std::to_string(step));
            print_img_in_csv(ref_result, ref_filename.c_str(), Nx, Ny, n_grains);
            delete [] result;
        }

        /*
    if (step % 50 == 0) {
      one_step.hash_t.print_hash_table_v2();
      one_step.inv.check_from_dfslist(Nx*Ny);
    }
    */
    }

    //time_t end_time;
    //time(&end_time);
    clock_gettime(CLOCK_MONOTONIC, &tend);

    // valueType *end_result = one_step.decode_to_img();

    // print_img_in_csv(end_result, args->output.c_str(), Nx, Ny, n_grains);

    // // print all buckets
    // one_step.print_PNBuckets_to_file(args->bucket_output.c_str());

    delete args;
    // delete[] mtx;
    // delete[] end_result;

    //printf("running_time=%lf\n", difftime(end_time, start_time));
    printf("running_time=%lf\n", ((double)tend.tv_sec + 1.0e-9 * tend.tv_nsec) -
                                     ((double)tstart.tv_sec + 1.0e-9 * tstart.tv_nsec));

    printf("Have a nice day!\n");
    return 0;
}
